/*
** FAAD2 - Freeware Advanced Audio (AAC) Decoder including SBR decoding
** Copyright (C) 2003-2004 M. Bakker, Ahead Software AG, http://www.nero.com
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
**
** Any non-GPL usage of this software or parts of this software is strictly
** forbidden.
**
** Commercial non-GPL licensing of this software is possible.
** For more info contact Ahead Software through Mpeg4AAClicense@nero.com.
**
** $Id: mp4atom.c,v 1.23 2011/05/16 00:00:00 PDSoft Exp $
**/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <math.h>
#include "mp4ff.h"
#include "mpxplay.h"
#include "newfunc\newfunc.h"

//#define MP4FF_DEBUG 1

typedef struct atom_description_s{
 uint8_t atom_id;
 char atomname[4];
 //int32_t (*readatom)(mp4ff_t *f, int32_t size, uint8_t atom_id);
}atom_description_s;

static struct atom_description_s atom_descriptions[]=
{
 {ATOM_MOOV,  "moov"},
 {ATOM_TRAK,  "trak"},
 {ATOM_EDTS,  "edts"},
 {ATOM_MDIA,  "mdia"},
 {ATOM_MINF,  "minf"},
 {ATOM_STBL,  "stbl"},
 {ATOM_UDTA,  "udta"},
 {ATOM_ILST,  "ilst"},
 {ATOM_TITLE, "nam"},
 {ATOM_ARTIST,"ART"},
 {ATOM_WRITER,"wrt"},
 {ATOM_ALBUM, "alb"},
 {ATOM_DATE,  "day"},
 {ATOM_TOOL,  "too"},
 {ATOM_COMMENT,"cmt"},
 {ATOM_GENRE1,"gen"},
 {ATOM_TRACK, "trkn"},
 {ATOM_DISC,  "disk"},
 {ATOM_COMPILATION,"cpil"},
 {ATOM_GENRE2,"gnre"},
 {ATOM_TEMPO, "tmpo"},
 {ATOM_COVER, "covr"},
 {ATOM_DRMS,  "drms"},
 {ATOM_SINF,  "sinf"},
 {ATOM_SCHI,  "schi"},

 {ATOM_FTYP,"ftyp"},
 {ATOM_MDAT,"mdat"},
 {ATOM_MVHD,"mvhd"},
 {ATOM_TKHD,"tkhd"},
 {ATOM_TREF,"tref"},
 {ATOM_MDHD,"mdhd"},
 {ATOM_VMHD,"vmhd"},
 {ATOM_SMHD,"smhd"},
 {ATOM_HMHD,"hmhd"},
 {ATOM_STSD,"stsd"},
 {ATOM_STTS,"stts"},
 {ATOM_STSZ,"stsz"},
 {ATOM_STZ2,"stz2"},
 {ATOM_STCO,"stco"},
 {ATOM_CO64,"co64"},
 {ATOM_STSC,"stsc"},
 {ATOM_MP4A,"mp4a"},
 {ATOM_MP4V,"mp4v"},
 {ATOM_MP4S,"mp4s"},
 {ATOM_ESDS,"esds"},
 {ATOM_META,"meta"},
 {ATOM_NAME,"name"},
 {ATOM_DATA,"data"},
 {ATOM_CTTS,"ctts"},
 {ATOM_FRMA,"frma"},
 {ATOM_IVIV,"iviv"},
 {ATOM_PRIV,"priv"},
 {ATOM_USER,"user"},
 {ATOM_KEY ,"key "},
 {ATOM_WIDE,"wide"},
 {ATOM_ALAC,"alac"},

 {ATOM_UNKNOWN,""}
};

static uint8_t mp4ff_atom_name_to_type(uint8_t *cname)
{
 struct atom_description_s *a=&atom_descriptions[0];
 const uint32_t iname=(*(uint32_t *)cname);
 do{
  if((*(uint32_t *)&a->atomname[0])==iname)
   break;
  a++;
 }while(a->atom_id<ATOM_UNKNOWN);
 return a->atom_id;
}

// read atom header, return atom size, atom size is with header included
uint64_t mp4ff_atom_read_header(mp4ff_t *f, uint8_t *atom_type, uint8_t *header_size, uint32_t *atom_name)
{
 uint8_t atom_header[4];
 uint64_t size=0;

 *atom_type=0;
 *header_size=0;

 if(mp4ff_read_data(f, (int8_t *)(&atom_header[0]), 4)!=4){
#ifdef MP4FF_DEBUG
  fprintf(stdout,"EOF\n");
#endif
  goto err_out_arh;
 }

 size=PDS_GETB_BE32(atom_header);
 if(!size)
  goto err_out_arh;

 if(mp4ff_read_data(f, (int8_t *)(&atom_header[0]), 4)!=4){
#ifdef MP4FF_DEBUG
  fprintf(stdout,"EOF\n");
#endif
  goto err_out_arh;
 }

 if(atom_name)
  *atom_name=*((uint32_t *)&atom_header[0]);

 *header_size = 8;

 // check for 64 bit atom size
 if(size == 1){
  *header_size = 16;
  size = mp4ff_read_int64(f);
 }

 *atom_type = mp4ff_atom_name_to_type(&atom_header[0]);

#ifdef MP4FF_DEBUG
 fprintf(stdout,"a:%4.4s hs:%d s:%lld pos:%d %X\n",&atom_header[0],(int)(*header_size),size,(long)mp4ff_position(f)-8,(long)mp4ff_position(f)-8);
#endif

err_out_arh:
 return size;
}

static int32_t mp4ff_read_mdhd(mp4ff_t *f)
{
 mp4ff_track_t * p_track = f->lasttrack;
 uint32_t version;

 version = mp4ff_read_int32(f);
 if(version==1){
  mp4ff_read_int64(f);//creation-time
  mp4ff_read_int64(f);//modification-time
  p_track->timeScale = mp4ff_read_int32(f);
  p_track->duration  = mp4ff_read_int64(f);
 }else{ //version == 0
  uint32_t temp;

  mp4ff_read_int32(f);//creation-time
  mp4ff_read_int32(f);//modification-time

  p_track->timeScale = mp4ff_read_int32(f);
  temp = mp4ff_read_int32(f);
  p_track->duration = (temp == (uint32_t)(-1)) ? (uint64_t)(-1) : (uint64_t)(temp);
 }

 mp4ff_read_int16(f);
 mp4ff_read_int16(f);
 return 1;
}

static int32_t mp4ff_read_esds(mp4ff_t *f)
{
 mp4ff_track_t *p_track = f->lasttrack;
 uint8_t tag;
 uint32_t temp;

 mp4ff_read_char(f);  // version
 mp4ff_read_int24(f); // flags

 // get and verify ES_DescrTag
 tag = mp4ff_read_char(f);
 if(tag==0x03){
  if(mp4ff_read_mp4_descr_length(f) < (5 + 15)) // ???
   return 1;
  mp4ff_read_int24(f); // skip 3 bytes
 }else
  mp4ff_read_int16(f); // skip 2 bytes

 // get and verify DecoderConfigDescrTab
 if(mp4ff_read_char(f) != 0x04)
  return 1;

 temp = mp4ff_read_mp4_descr_length(f);
 if(temp<13)
  return 1;

 p_track->audioType  = mp4ff_read_char(f);
 mp4ff_read_int32(f);//0x15000414 ????
 p_track->maxBitrate = mp4ff_read_int32(f);
 p_track->avgBitrate = mp4ff_read_int32(f);

 // get and verify DecSpecificInfoTag
 if(mp4ff_read_char(f) != 0x05)
  return 1;

 p_track->decoderConfigLen = mp4ff_read_mp4_descr_length(f);

 if(p_track->decoderConfig)
  free(p_track->decoderConfig);
 p_track->decoderConfig = malloc(p_track->decoderConfigLen);
 if(p_track->decoderConfig)
  mp4ff_read_data(f, (int8_t *)p_track->decoderConfig, p_track->decoderConfigLen);
 else
  p_track->decoderConfigLen = 0;

 return 0;
}

static int32_t mp4ff_read_mp4a(mp4ff_t *f)
{
 mp4ff_track_t *p_track = f->lasttrack;
 uint32_t version;
 uint8_t atom_type,header_size;

 mp4ff_read_int16(f); // reserved
 mp4ff_read_int32(f); // reserved

 mp4ff_read_int16(f); // data_reference_index
 version=mp4ff_read_int16(f);
 mp4ff_read_int16(f); // revision
 mp4ff_read_int32(f); // vendor_id

 p_track->channelCount = mp4ff_read_int16(f);
 p_track->sampleSize   = mp4ff_read_int16(f);
 mp4ff_read_int16(f); // compression id
 p_track->sampleRate   = mp4ff_read_int32(f); // ???  2 bytes packet_size==0 + 2 bytes sample_rate
 mp4ff_read_int16(f); // unknown

 if(version){
  mpxp_filesize_t pos=mp4ff_position(f);
  if(version==1)
   pos+=48;
  else
   pos+=68;
  mp4ff_set_position(f,pos);
 }

 if(mp4ff_atom_read_header(f, &atom_type, &header_size, NULL)>0)
  if(atom_type == ATOM_ESDS)
   mp4ff_read_esds(f);

 return 0;
}

static uint32_t mp4ff_read_alac(mp4ff_t *f,int32_t size)
{
 mp4ff_track_t *p_track = f->lasttrack;
 uint16_t version,i;

 size-=8;

 for (i = 0; i < 6; i++)
  mp4ff_read_char(f); // reserved
 size-=6;

 version=mp4ff_read_int16(f); size-=2; // version have to be 1
 mp4ff_read_int16(f); // revision level
 mp4ff_read_int32(f); // vendor
 mp4ff_read_int16(f); // something
 size-=8;

 p_track->channelCount = mp4ff_read_int16(f); size-=2;
 p_track->sampleSize   = mp4ff_read_int16(f); size-=2;
 mp4ff_read_int16(f); // compression id
 mp4ff_read_int16(f); // packet_size
 size-=4;
 p_track->sampleRate   = mp4ff_read_int16(f); size-=2;
 mp4ff_read_int16(f); size-=2;

 p_track->decoderConfigLen = size+12+8;
 if(p_track->decoderConfig)
  free(p_track->decoderConfig);
 p_track->decoderConfig = malloc(p_track->decoderConfigLen);
 if(p_track->decoderConfig){
  ((uint32_t *)p_track->decoderConfig)[0]=0x0c000000;
  memcpy(p_track->decoderConfig+4,"frma",4);
  memcpy(p_track->decoderConfig+8,"alac",4);
  mp4ff_read_data(f,(int8_t *)p_track->decoderConfig+12,size);
  memset(p_track->decoderConfig+12+size,0,8);
 }else
  p_track->decoderConfigLen = 0;

 p_track->audioType=MP4_ALAC_AUDIO_TYPE;

 return 0;
}

static double ff_int2dbl(int64_t v)
{
 if(v+v > 0xFFEULL<<52)
  return 0.0;
 return ldexp(((v&((1LL<<52)-1)) + (1LL<<52)) * (v>>63|1), (v>>52&0x7FF)-1075);
}

static struct audio_atom_s{
 uint32_t atom_name;
 uint32_t wave_id;
}audio_atoms[]={
 { PDS_GET4C_LE32('i', 'n', '3', '2'),MPXPLAY_WAVEID_PCM_SBE },
 //{ PDS_GET4C_LE32('i', 'n', '3', '2'),MPXPLAY_WAVEID_PCM_SLE },
 { PDS_GET4C_LE32('i', 'n', '2', '4'),MPXPLAY_WAVEID_PCM_SBE },
 //{ PDS_GET4C_LE32('i', 'n', '2', '4'),MPXPLAY_WAVEID_PCM_SLE },
 { PDS_GET4C_LE32('t', 'w', 'o', 's'),MPXPLAY_WAVEID_PCM_SBE },
 { PDS_GET4C_LE32('s', 'o', 'w', 't'),MPXPLAY_WAVEID_PCM_SLE },
 { PDS_GET4C_LE32('f', 'l', '3', '2'),MPXPLAY_WAVEID_PCM_F32BE },
 //{ PDS_GET4C_LE32('f', 'l', '3', '2'),MPXPLAY_WAVEID_PCM_FLOAT },
 { PDS_GET4C_LE32('f', 'l', '6', '4'),MPXPLAY_WAVEID_PCM_F64BE },
 //{ PDS_GET4C_LE32('f', 'l', '6', '4'),MPXPLAY_WAVEID_PCM_F64LE },
 { PDS_GET4C_LE32('r', 'a', 'w', ' '),MPXPLAY_WAVEID_PCM_ULE },
 { PDS_GET4C_LE32('N', 'O', 'N', 'E'),MPXPLAY_WAVEID_PCM_ULE },

 { PDS_GET4C_LE32('.', 'm', 'p', '2'),MPXPLAY_WAVEID_MP2 },
 { PDS_GET4C_LE32('.', 'm', 'p', '3'),MPXPLAY_WAVEID_MP3 },
 { 0x6D730055                        ,MPXPLAY_WAVEID_MP3 },
 { PDS_GET4C_LE32('a', 'c', '-', '3'),MPXPLAY_WAVEID_AC3 },
 { PDS_GET4C_LE32('s', 'a', 'c', '3'),MPXPLAY_WAVEID_AC3 },
 { PDS_GET4C_LE32('W', 'M', 'A', '2'),MPXPLAY_WAVEID_WMAV2 },

 { PDS_GET4C_LE32('a', 'l', 'a', 'w'),PDS_GET4C_LE32('A', 'L', 'A', 'W') },
 { PDS_GET4C_LE32('u', 'l', 'a', 'w'),PDS_GET4C_LE32('U', 'L', 'A', 'W') },
 { PDS_GET4C_LE32('i', 'm', 'a', '4'),PDS_GET4C_LE32('I', 'M', 'A', '4') },
 { PDS_GET4C_LE32('M', 'A', 'C', '3'),PDS_GET4C_LE32('M', 'A', 'C', '3') },
 { PDS_GET4C_LE32('M', 'A', 'C', '6'),PDS_GET4C_LE32('M', 'A', 'C', '6') },
 { PDS_GET4C_LE32('.', 'm', 'p', '1'),PDS_GET4C_LE32('M', 'P', '1', ' ') },
 { PDS_GET4C_LE32('s', 'a', 'm', 'r'),PDS_GET4C_LE32('A', 'M', 'R', 'N') },
 { PDS_GET4C_LE32('s', 'a', 'w', 'b'),PDS_GET4C_LE32('A', 'M', 'R', 'W') },
 { PDS_GET4C_LE32('a', 'g', 's', 'm'),PDS_GET4C_LE32('A', 'G', 'S', 'M') },
 { PDS_GET4C_LE32('Q', 'c', 'l', 'p'),PDS_GET4C_LE32('Q', 'C', 'L', 'P') },
 { PDS_GET4C_LE32('Q', 'c', 'l', 'q'),PDS_GET4C_LE32('Q', 'C', 'L', 'Q') },
 { PDS_GET4C_LE32('s', 'q', 'c', 'p'),PDS_GET4C_LE32('S', 'Q', 'C', 'P') },
 { PDS_GET4C_LE32('Q', 'D', 'M', 'C'),PDS_GET4C_LE32('Q', 'D', 'M', 'C') },
 { PDS_GET4C_LE32('Q', 'D', 'M', '2'),PDS_GET4C_LE32('Q', 'D', 'M', '2') },
 { PDS_GET4C_LE32('v', 'd', 'v', 'a'),PDS_GET4C_LE32('V', 'D', 'V', 'A') },
 { PDS_GET4C_LE32('d', 'v', 'c', 'a'),PDS_GET4C_LE32('D', 'V', 'C', 'A') },
 /*{ PDS_GET4C_LE32('a', 'l', 'a', 'w'),MPXPLAY_WAVEID_UNSUPPORTED },
 { PDS_GET4C_LE32('u', 'l', 'a', 'w'),MPXPLAY_WAVEID_UNSUPPORTED },
 { PDS_GET4C_LE32('i', 'm', 'a', '4'),MPXPLAY_WAVEID_UNSUPPORTED },
 { PDS_GET4C_LE32('M', 'A', 'C', '3'),MPXPLAY_WAVEID_UNSUPPORTED },
 { PDS_GET4C_LE32('M', 'A', 'C', '6'),MPXPLAY_WAVEID_UNSUPPORTED },
 { PDS_GET4C_LE32('.', 'm', 'p', '1'),MPXPLAY_WAVEID_UNSUPPORTED },
 { PDS_GET4C_LE32('s', 'a', 'm', 'r'),MPXPLAY_WAVEID_UNSUPPORTED },
 { PDS_GET4C_LE32('s', 'a', 'w', 'b'),MPXPLAY_WAVEID_UNSUPPORTED },
 { PDS_GET4C_LE32('a', 'g', 's', 'm'),MPXPLAY_WAVEID_UNSUPPORTED },
 { PDS_GET4C_LE32('Q', 'c', 'l', 'p'),MPXPLAY_WAVEID_UNSUPPORTED },
 { PDS_GET4C_LE32('Q', 'c', 'l', 'q'),MPXPLAY_WAVEID_UNSUPPORTED },
 { PDS_GET4C_LE32('s', 'q', 'c', 'p'),MPXPLAY_WAVEID_UNSUPPORTED },
 { PDS_GET4C_LE32('Q', 'D', 'M', 'C'),MPXPLAY_WAVEID_UNSUPPORTED },
 { PDS_GET4C_LE32('Q', 'D', 'M', '2'),MPXPLAY_WAVEID_UNSUPPORTED },
 { PDS_GET4C_LE32('v', 'd', 'v', 'a'),MPXPLAY_WAVEID_UNSUPPORTED },
 { PDS_GET4C_LE32('d', 'v', 'c', 'a'),MPXPLAY_WAVEID_UNSUPPORTED },*/
 { 0, 0 }
};

static int32_t mp4ff_read_audio(mp4ff_t *f, uint32_t atom_name, uint32_t size)
{
 uint32_t version,flags;
 mp4ff_track_t *p_track = f->lasttrack;
 struct audio_atom_s *ats=&audio_atoms[0];

 if(size<28)
  return -1;

 do{
  if(atom_name==ats->atom_name){
   p_track->wave_id=ats->wave_id;
   break;
  }
  ats++;
 }while(ats->atom_name);

 if(!p_track->wave_id)
  if((atom_name&0xFFFF)==((uint32_t)'m'+((uint32_t)'s'<<8)) || (atom_name&0xFFFF) == ((uint32_t)'T'+((uint32_t)'S'<<8)))
   p_track->wave_id=((atom_name>>8)&0xff00)|(atom_name>>24);

 if(!p_track->wave_id){
  p_track->type=TRACK_UNKNOWN;
  return -1;
 }

 p_track->type=TRACK_AUDIO;

 mp4ff_read_int32(f); // reserved
 mp4ff_read_int16(f); // reserved
 mp4ff_read_int16(f); // data_reference_index

 version=mp4ff_read_int16(f);
 mp4ff_read_int16(f); // revision level
 mp4ff_read_int32(f); // vendor

 p_track->channelCount = mp4ff_read_int16(f);
 p_track->sampleSize   = mp4ff_read_int16(f);
 mp4ff_read_int16(f); // cid
 mp4ff_read_int16(f); // packet_size
 p_track->sampleRate   = mp4ff_read_int32(f)>>16;

 switch(version){
  case 1:
   if(size>=(28+16)){
    p_track->samples_per_frame=mp4ff_read_int32(f); // samples_per_frame
    mp4ff_read_int32(f); // bytes_per_packet
    p_track->bytes_per_frame=mp4ff_read_int32(f); // bytes_per_frame
    mp4ff_read_int32(f); // bytes_per_sample
   }
   break;
  case 2:
   if(size>=(28+32)){
    p_track->sampleRate = ff_int2dbl(mp4ff_read_int64(f));
    p_track->channelCount = mp4ff_read_int32(f);
    mp4ff_read_int32(f); // always 0x7F000000 ?
    p_track->sampleSize   = mp4ff_read_int32(f);
    flags=mp4ff_read_int32(f);
    p_track->bytes_per_frame=mp4ff_read_int32(f);
    p_track->samples_per_frame=mp4ff_read_int32(f);
   }
   break;
 }

 // ??? read ESDS (for WMA)

 return 0;
}

static int32_t mp4ff_read_stsd(mp4ff_t *f)
{
 uint32_t i,atom_name;
 mp4ff_track_t *p_track = f->lasttrack;
 uint8_t atom_type,header_size;

 mp4ff_read_int32(f); // version (1) & flags (3)

 p_track->stsd_entry_count = mp4ff_read_int32(f);

 for(i=0; i<p_track->stsd_entry_count; i++){
  uint64_t size = mp4ff_atom_read_header(f, &atom_type, &header_size, &atom_name);
  uint64_t skip = mp4ff_position(f)+size;

  switch(atom_type){
   case ATOM_MP4A:p_track->type=TRACK_AUDIO;mp4ff_read_mp4a(f);break;
   case ATOM_MP4V:p_track->type=TRACK_VIDEO;break;
   case ATOM_MP4S:p_track->type=TRACK_SYSTEM;break;
   case ATOM_ALAC:p_track->type=TRACK_AUDIO;mp4ff_read_alac(f,size);break;
   default:mp4ff_read_audio(f,atom_name,size);
  }

  mp4ff_set_position(f, skip);
 }

 return 0;
}

static int32_t mp4ff_read_ctts(mp4ff_t *f)
{
 int32_t i;
 mp4ff_track_t *p_track = f->lasttrack;

 if(p_track->ctts_entry_count)
  return 0;

 mp4ff_read_int32(f); // version (1) & flags (3)

 p_track->ctts_entry_count = mp4ff_read_int32(f);
 if(!p_track->ctts_entry_count)
  return 1;

 p_track->ctts_sample_count  = (int32_t*)malloc(p_track->ctts_entry_count * sizeof(int32_t));
 p_track->ctts_sample_offset = (int32_t*)malloc(p_track->ctts_entry_count * sizeof(int32_t));
 if(!p_track->ctts_sample_count || !p_track->ctts_sample_offset){
  p_track->ctts_entry_count = 0;
  return 1;
 }

 for(i=0; i<f->lasttrack->ctts_entry_count; i++){
  p_track->ctts_sample_count[i]  = mp4ff_read_int32(f);
  p_track->ctts_sample_offset[i] = mp4ff_read_int32(f);
 }

 return 0;
}

static int32_t mp4ff_read_stts(mp4ff_t *f)
{
 int32_t i;
 mp4ff_track_t * p_track = f->lasttrack;

 if(p_track->stts_entry_count)
  return 0;

 mp4ff_read_int32(f); // version (1) & flags (3)

 p_track->stts_entry_count = mp4ff_read_int32(f);
 if(!p_track->stts_entry_count)
  return 1;

 p_track->stts_sample_count = (int32_t*)malloc(p_track->stts_entry_count * sizeof(int32_t));
 p_track->stts_sample_delta = (int32_t*)malloc(p_track->stts_entry_count * sizeof(int32_t));
 if(!p_track->stts_sample_count || !p_track->stts_sample_delta){
  p_track->stts_entry_count = 0;
  return 1;
 }

 for(i=0; i<p_track->stts_entry_count; i++){
  p_track->stts_sample_count[i] = mp4ff_read_int32(f);
  p_track->stts_sample_delta[i] = mp4ff_read_int32(f);
 }

#ifdef MP4FF_DEBUG
 fprintf(stdout,"STTS ec:%d sc0:%d sd0:%d\n",p_track->stts_entry_count,
  p_track->stts_sample_count[0],p_track->stts_sample_delta[0]);
#endif

 return 0;
}

static int32_t mp4ff_read_stsc(mp4ff_t *f)
{
 int32_t i;
 mp4ff_track_t *p_track = f->lasttrack;

 if(p_track->stsc_entry_count)
  return 0;

 mp4ff_read_int32(f); // version (1) & flags (3)

 p_track->stsc_entry_count = mp4ff_read_int32(f);
 if(!p_track->stsc_entry_count)
  return 1;

 p_track->stsc_first_chunk       = (int32_t*)malloc(p_track->stsc_entry_count*sizeof(int32_t));
 p_track->stsc_samples_per_chunk = (int32_t*)malloc(p_track->stsc_entry_count*sizeof(int32_t));
 p_track->stsc_sample_desc_index = (int32_t*)malloc(p_track->stsc_entry_count*sizeof(int32_t));

 if(!p_track->stsc_first_chunk || !p_track->stsc_samples_per_chunk || !p_track->stsc_sample_desc_index)
  return 1;

 for(i=0; i<p_track->stsc_entry_count; i++){
  p_track->stsc_first_chunk[i]       = mp4ff_read_int32(f);
  p_track->stsc_samples_per_chunk[i] = mp4ff_read_int32(f);
  p_track->stsc_sample_desc_index[i] = mp4ff_read_int32(f);
 }

#ifdef MP4FF_DEBUG
 fprintf(stdout,"STSC ec:%d fc0:%d spc0:%d fc1:%d spc1:%d\n",p_track->stsc_entry_count,
  p_track->stsc_first_chunk[0],p_track->stsc_samples_per_chunk[0],
  p_track->stsc_first_chunk[1],p_track->stsc_samples_per_chunk[1]);
#endif

 return 0;
}

static int32_t mp4ff_read_stsz(mp4ff_t *f)
{
 mp4ff_track_t *p_track = f->lasttrack;

 mp4ff_read_int32(f); // version (1) & flags (3)

 p_track->stsz_sample_size = mp4ff_read_int32(f);
 p_track->stsz_max_sample_size = p_track->stsz_sample_size;
 p_track->stsz_sample_count = mp4ff_read_int32(f);

 if(!p_track->stsz_sample_size && p_track->stsz_sample_count){
  int32_t i;

  p_track->stsz_table = (int32_t*)malloc(p_track->stsz_sample_count*sizeof(int32_t));
  if(!p_track->stsz_table)
   return 1;

  for(i=0; i<p_track->stsz_sample_count; i++){
   int32_t sample_size = mp4ff_read_int32(f);
   p_track->stsz_table[i] = sample_size;
   if(sample_size > p_track->stsz_max_sample_size)
    p_track->stsz_max_sample_size=sample_size;
  }
 }

#ifdef MP4FF_DEBUG
 fprintf(stdout,"STSZ ss:%d mss:%d sc:%d\n",p_track->stsz_sample_size,p_track->stsz_max_sample_size,p_track->stsz_sample_count);
#endif

 return 0;
}

static int32_t mp4ff_read_stco(mp4ff_t *f, const uint8_t atom_type)
{
 mp4ff_track_t *p_track = f->lasttrack;
 mpxp_filesize_t *scop;
 uint32_t i;

 if(p_track->stco_entry_count)
  return 0;

 mp4ff_read_int32(f); // version (1) & flags (3)

 i = mp4ff_read_int32(f);
 if(!i)
  return 1;
 p_track->stco_entry_count = i;

 scop = (mpxp_filesize_t*)malloc(i*sizeof(mpxp_filesize_t));
 if(!scop)
  return 1;
 p_track->stco_chunk_offset = scop;

#ifdef MPXPLAY_FSIZE64
 if(atom_type==ATOM_CO64){
  do{
   *scop++ = mp4ff_read_int64(f);
  }while(--i);
 }else
#endif
  do{
   *scop++ = mp4ff_read_int32(f);
  }while(--i);

#ifdef MP4FF_DEBUG
 fprintf(stdout,"STCO ec:%d co0:%lld co1:%lld co2:%lld\n",p_track->stco_entry_count,
  (int64_t)p_track->stco_chunk_offset[0],(int64_t)p_track->stco_chunk_offset[1],(int64_t)p_track->stco_chunk_offset[2]);
#endif

 return 0;
}

static int32_t mp4ff_read_meta(mp4ff_t *f, const uint64_t size)
{
 uint64_t subsize, sumsize = 0;
 uint8_t atom_type;
 uint8_t header_size = 0;

 mp4ff_read_int32(f); // version (1) & flags (3)

 while(sumsize < (size-(header_size+4))){
  subsize = mp4ff_atom_read_header(f, &atom_type, &header_size, NULL);
  if(subsize <= header_size+4)
   return 1;

  if(atom_type == ATOM_ILST)
   mp4ff_parse_metadata(f, (uint32_t)(subsize-(header_size+4)));
  else
   mp4ff_set_position(f, mp4ff_position(f)+subsize-header_size);

  sumsize += subsize;
 }

 return 0;
}

int32_t mp4ff_atom_read(mp4ff_t *f, const int32_t size, const uint8_t atom_type)
{
 uint64_t dest_position = mp4ff_position(f)+size-8;

 if(f->openmode&MP4FF_OPENMODE_FULLLOAD){
  switch(atom_type){
   case ATOM_STSZ: mp4ff_read_stsz(f);break; // sample size box
   case ATOM_STTS: mp4ff_read_stts(f);break; // time to sample box
   case ATOM_CTTS: mp4ff_read_ctts(f);break; // composition offset box
   case ATOM_STSC: mp4ff_read_stsc(f);break; // sample to chunk box
#ifdef MPXPLAY_FSIZE64
   case ATOM_CO64:
#endif
   case ATOM_STCO: mp4ff_read_stco(f,atom_type);break; // chunk offset box
  }
 }
 switch(atom_type){
  case ATOM_STSD: mp4ff_read_stsd(f);break; // sample description box
  case ATOM_MDHD: mp4ff_read_mdhd(f);break; // track header
  case ATOM_META: mp4ff_read_meta(f,size);break;// iTunes Metadata box
 }

 return mp4ff_set_position(f, dest_position);
}
