//**************************************************************************
//*                     This file is part of the                           *
//*             AudioCV - a general audio converter program                *
//*                  The source code of AudioCV is                         *
//*          (c) copyright 2001-2004 by PDSoft (Attila Padar)              *
//*                    http://mpxplay.cjb.net                              *
//* email: mpxplay@freemail.hu (please write AudioCV in the subject field) *
//**************************************************************************
//wav file input/output functions

#include "audiocv.h"

#define WAVID_RIFF    0x46464952 // RIFF (FIRR)
#define WAVID_WAVE    0x45564157 // WAVE (EVAW)
#define WAVID_fmt     0x20746d66 // fmt  ( tmf)
#define WAVID_data    0x61746164 // data (atad)

extern long acv_filelen(FILE *);
//--------------------------------------------------------------------------
       int wav_readheader(acv_fileinfo *);
static unsigned int  wav_chunk_search(acv_fileinfo *,unsigned long);
static unsigned long readlong_fromfile(acv_fileinfo *);
       void wav_readdata(acv_fileinfo *,unsigned int);
//--------------------------------------------------------------------------
       int  wav_writeheader(acv_fileinfo *);
       int  wav_writedata(acv_fileinfo *);
//--------------------------------------------------------------------------
//wav input routines
int wav_readheader(acv_fileinfo *af_in)
{
 struct RIFF{
  unsigned long riffID;
  unsigned long rLen;
 }riff;
 struct WAVE {
  unsigned long waveID;
 }wave;
 struct FORMAT {
  unsigned long fLen;
  unsigned short wTag;
  unsigned short wChannel;
  unsigned long nSample;
  unsigned long nByte;
  unsigned short align;
  unsigned short sample;
 }fmt;

 unsigned long indatalen,inallsamplenum=0,maxdatalen;

 fseek(af_in->fp,0,SEEK_SET);
 if(fread((char *)(&riff),1,sizeof(struct RIFF),af_in->fp)!=sizeof(struct RIFF))
  return -1;
 if(riff.riffID!=WAVID_RIFF)
  return -1;
 if(fread((char *)(&wave),1,sizeof(struct WAVE),af_in->fp)!=sizeof(struct WAVE))
  return -1;
 if(wave.waveID!=WAVID_WAVE)  // RIFF....WAVE           header
  return -1;
 if(!wav_chunk_search(af_in,WAVID_fmt))    // search for 'fmt' chunk
  return -1;
 fread((char *)(&fmt),1,sizeof(struct FORMAT),af_in->fp); // read 'fmt' chunk
 if(fmt.fLen<16)               // too short fmt chunk
  return -1;
 if(!fmt.wChannel)             // channels==0 ?
  return -1;
 if(!fmt.sample || fmt.sample>PCM_MAX_BITS) //bits
  return -1;
 if( ( ( (fmt.sample+7) >>3 )*fmt.wChannel)!=fmt.align)  // (bits+7)/8*channels!=bytespersample
  return -1;

 if(readlong_fromfile(af_in)!=WAVID_data){ // first (and fast) search for 'data' chunk
  fseek(af_in->fp,fmt.fLen-20,SEEK_CUR);
  if(!wav_chunk_search(af_in,WAVID_data))    // (slow and valid) search for 'data' chunk
   return -1;
 }
 indatalen=readlong_fromfile(af_in);
 maxdatalen=acv_filelen(af_in->fp)-ftell(af_in->fp); // maxdatalen=filesize-headersize
 if(!indatalen || (indatalen>maxdatalen))  // indatalen have to be less than maxdatalen
  indatalen=maxdatalen;
 inallsamplenum=indatalen/(unsigned long)fmt.align*(unsigned long)fmt.wChannel;
 if(inallsamplenum<fmt.wChannel)              // we need 1 sample at least
  return -1;

 af_in->allsamplenum=inallsamplenum;
 af_in->channels=fmt.wChannel;   // number of channels
 af_in->freq    =fmt.nSample;    // frequency;
 af_in->databits=fmt.sample;     // bits
 af_in->bytespersample=fmt.align/fmt.wChannel;

 switch(fmt.wTag){
  // integer based uncompressed PCM
  case 0x01:af_in->scalebits=af_in->bytespersample*8;
	    af_in->filemode&=~CF_FLOAT;
	    break;
  // float (32-bit IEEE) based uncompressed PCM
  case 0x03:af_in->scalebits=BITTYPE_UNDEFINED;
	    af_in->filemode|=CF_FLOAT;
	    break;
  default:return -1;
 }
 af_in->filetype=FT_WAV;
 return 0;
}

static unsigned int wav_chunk_search(acv_fileinfo *af_in,unsigned long search_id)
{
 long jump,flen=acv_filelen(af_in->fp);
 unsigned int retcode=0,counter=10;
 do{
  if(readlong_fromfile(af_in)==search_id){
   retcode=1;
   break;
  }
  if(feof(af_in->fp))
   break;
  jump=readlong_fromfile(af_in);
  if(jump<0 || (jump>=(flen-ftell(af_in->fp))))
   break;
  fseek(af_in->fp,jump,SEEK_CUR);
 }while(counter--);
 return retcode;
}

static unsigned long readlong_fromfile(acv_fileinfo *af_in)
{
 unsigned long x;
 fread((char *)(&x),1,4,af_in->fp);
 return x;
}

/*#define SWAP_4BYTES(buf) (((buf)[0]<<24)|((buf)[1]<<16)|((buf)[2]<<8)|((buf)[3]&0xff))

static void acv_block_4byteswap(char *ptr,unsigned int samplenum)
{
 while(samplenum--){
  *((unsigned long *)ptr)=SWAP_4BYTES(ptr);
  ptr+=4;
 }
}*/

void wav_readdata(acv_fileinfo *af_in,unsigned int numsamples)
{
 numsamples*=af_in->channels;
 af_in->blocksamplenum=fread(af_in->buffer,af_in->bytespersample,numsamples,af_in->fp);
 if((af_in->currsamplenum+af_in->blocksamplenum)>af_in->allsamplenum)
  af_in->blocksamplenum=af_in->allsamplenum-af_in->currsamplenum;
}

//-----------------------------------------------------------------------
// wav out functions
int wav_writeheader(acv_fileinfo *af_out)
{
 struct RIFF{
  unsigned long riffID;    //0x00
  unsigned long rLen;      //0x04
 }riff;
 struct WAVE{
  unsigned long waveID;    //0x08
 }wave;
 struct FORMAT{
  unsigned long fmtID;     //0x0c
  unsigned long fLen;      //0x10
  unsigned short wTag;     //0x14
  unsigned short wChannel; //0x16
  unsigned long nSample;   //0x18
  unsigned long nByte;     //0x1c
  unsigned short align;    //0x20
  unsigned short sample;   //0x22
 }fmt;
 struct DATA{
  unsigned long dataID;    //0x24
  unsigned long dLen;      //0x28
 }data;

 if(af_out->filemode&CF_FLOAT){
  fmt.wTag  = 0x0003; // 32-bit IEEE float WAV files
  af_out->databits=32;
  af_out->scalebits=BITTYPE_UNSCALED;
  af_out->bytespersample=4;
 }else{
  fmt.wTag  = 0x0001; // n-bit integer WAV files
  if(!af_out->databits)  // have to be set
   af_out->databits=BITTYPE_DEFAULT;
  af_out->scalebits=(af_out->databits+7)&0xfffffff8; // filebits (rounding up to mod 8)
  af_out->bytespersample=af_out->scalebits/8;
 }

 riff.riffID=WAVID_RIFF;
 riff.rLen  =acv_filelen(af_out->fp)-sizeof(struct RIFF);

 wave.waveID=WAVID_WAVE;

 fmt.fmtID   =WAVID_fmt;
 fmt.fLen    =sizeof(struct FORMAT)-8;

 fmt.wChannel=af_out->channels;
 fmt.nSample =af_out->freq;
 fmt.nByte   =af_out->freq*af_out->channels*af_out->bytespersample;
 fmt.align   =af_out->channels*af_out->bytespersample;
 fmt.sample  =af_out->databits;

 data.dataID=WAVID_data;
 data.dLen  =riff.rLen-(sizeof(struct WAVE)+sizeof(struct FORMAT)+sizeof(struct DATA));

 fwrite(&riff,1,sizeof(struct RIFF)  ,af_out->fp);
 fwrite(&wave,1,sizeof(struct WAVE)  ,af_out->fp);
 fwrite(&fmt ,1,sizeof(struct FORMAT),af_out->fp);
 fwrite(&data,1,sizeof(struct DATA)  ,af_out->fp);
 return 0;
}

int wav_writedata(acv_fileinfo *af_out)
{
 //if(af_out->filemode&CF_FLOAT)
 // acv_block_4byteswap(af_out->buffer,af_out->blocksamplenum);
 if(fwrite(af_out->buffer,af_out->bytespersample,af_out->blocksamplenum,af_out->fp)!=af_out->blocksamplenum)
  return ACV_ERROR_FILE_CANTWRITE;
 return 0;
}
