//**************************************************************************
//*                     This file is part of the                           *
//*             AudioCV - a general audio converter program                *
//*                  The source code of AudioCV is                         *
//*          (c) copyright 2001-2004 by PDSoft (Attila Padar)              *
//*                    http://mpxplay.cjb.net                              *
//* email: mpxplay@freemail.hu (please write AudioCV in the subject field) *
//**************************************************************************
//command line argument & encoding-listfile handling

#include "audiocv.h"
#include "utf8.h"
//---------------------------------------------------------------------------
extern void acv_change_extension_by_typenum(char *,unsigned int);
//---------------------------------------------------------------------------
       void get_commandline_options(int argc, char *argv[]);
       void set_variables_to_default(acv_fileinfo *,acv_fileinfo *);
       void accept_opt_filenames(acv_fileinfo *,acv_fileinfo *);
       void accept_opt_variables(acv_fileinfo *,acv_fileinfo *);
static void display_help(void);
//---------------------------------------------------------------------------

static char *tag_artist,*tag_title,*tag_album,*tag_date,*tag_genre,*tag_comment;
static char *freeopts[MAXFREEOPTS];
static int opt_conv_functions,opt_outbits,opt_outchannels,opt_outfreq;
static int opt_bitrate,opt_outfiletype;
static int disphelp;
static float opt_normdb,opt_normlimit;
#ifndef _WIN32
static char *opt_charset;
#endif

char *loadlistname;
int opt_functions,dispquiet;
int opt_tagenc_utf8;
int opt_channel_coupling,opt_const_bitrate;
int cv_clipslimit,cv_cutlimit,cv_mutelimit;
float cv_normdb,cv_normlimit,opt_lowpass_khz,opt_vbr_quality=-9.0f;

//option modes
#define ARG0     1     // enable option (-ct)
#define ARG1     2     // load argument (-br nnn, -ta string)
#define ARG2 (ARG0|ARG1) // enable option, load argument (-ctl n)
//argument types
#define ARGN     4     // numeric argument
#define ARGC     8     // character argument (possible string)
#define ARGF    16     // floating point argument
//bit switch
#define ARG_OR  32     // enable bit(s) in a variable
#define ARG_AND 64     // disable bit(s) in a variable

typedef struct {
 char *oname;
 int flags;
 void *var;
 int value;
 void *var2;
}topt;

static topt opts[] = {
{"br" ,ARG1|ARGN,         0,0,                    &opt_bitrate  },
{"ocp",ARG0,              &opt_channel_coupling,1,0},
{"ocb",ARG0,              &opt_const_bitrate,   1,0},
{"olp",ARG1|ARGF,         0,0,                    &opt_lowpass_khz},
{"oeq",ARG1|ARGF,         0,0,                    &opt_vbr_quality},

{"ta" ,ARG1|ARGC,         0,0,   &tag_artist   },
{"tt" ,ARG1|ARGC,         0,0,   &tag_title    },
{"tl" ,ARG1|ARGC,         0,0,   &tag_album    },
{"td" ,ARG1|ARGC,         0,0,   &tag_date     },
{"tg" ,ARG1|ARGC,         0,0,   &tag_genre    },
{"tc" ,ARG1|ARGC,         0,0,   &tag_comment  },

{"teu8",ARG0,             &opt_tagenc_utf8,1,0},
#ifndef _WIN32
{"tecp",ARG2|ARGC,        &opt_tagenc_utf8,1,&opt_charset},
#endif

{"ct" ,ARG0|ARG_OR,       &opt_conv_functions,CF_CUTZERO,  0},
{"ctl",ARG2|ARGN|ARG_OR,  &opt_conv_functions,CF_CUTZERO,  &cv_cutlimit},
{"mtl",ARG2|ARGN|ARG_OR,  &opt_conv_functions,CF_MUTEZERO, &cv_mutelimit},

{"na" ,ARG0|ARG_OR,       &opt_conv_functions,CF_ANALYZE,  0},
{"nr" ,ARG0|ARG_OR,       &opt_conv_functions,CF_ANALYZE|CF_NORMALIZE, 0},
{"ndb",ARG2|ARGF|ARG_OR,  &opt_conv_functions,CF_NORMALIZE,&opt_normdb},
{"ncl",ARG1|ARGN,         0,0,&cv_clipslimit},
{"nrl",ARG1|ARGF,         0,0,&opt_normlimit},

{"ob" ,ARG1|ARGN,         0,0,&opt_outbits},
{"of" ,ARG2|ARGN|ARG_OR,  &opt_conv_functions,CF_FREQ,     &opt_outfreq},
{"oc" ,ARG2|ARGN|ARG_OR,  &opt_conv_functions,CF_CHANNELS, &opt_outchannels},
{"sw" ,ARG0|ARG_OR,       &opt_conv_functions,CF_SWAPCHAN, 0},
{"obf",ARG0|ARG_OR,       &opt_conv_functions,CF_FLOAT,    0},

{"ll", ARG1|ARGC,         0,               0,     &loadlistname}, // for -na
{"llw",ARG2|ARGC,         &opt_outfiletype,FT_WAV,&loadlistname},
{"llo",ARG2|ARGC,         &opt_outfiletype,FT_OGG,&loadlistname},

{"idel",ARG0|ARG_OR,      &opt_functions,FUNC_DELETEINFILE,0},
{"iow" ,ARG0|ARG_OR,      &opt_functions,FUNC_OVERWRITE,0},

{"q"  ,ARG0,              &dispquiet,    1,0},
{"?"  ,ARG0,              &disphelp,     1,0},
{"h"  ,ARG0,              &disphelp,     1,0},
{0    ,0,0,0,0}
};

void get_commandline_options(int argc, char *argv[])
{
 int i,freeoptcount;
 topt *pointer;

 if(argc>1){
  freeopts[0]=&argv[0][0];
  freeoptcount=1;
  for(i=1;i<argc;i++){
   pointer=&opts[0];
   if(argv[i][0]=='-' || argv[i][0]=='/'){
    while(pointer->oname!=NULL){
     if(strcmp(&argv[i][1],pointer->oname)==0){
      if(pointer->flags&ARG0){
       if(pointer->flags&ARG_OR)
	*((int *) pointer->var)|=pointer->value;
       else
	if(pointer->flags&ARG_AND)
	 *((int *) pointer->var)&=pointer->value;
	else
	 *((int *) pointer->var)=pointer->value;
      }
      if(pointer->flags&ARG1){
       i++;
       if(i<argc /*&& argv[i][0]!='-'*/ && argv[i][0]!='/' && argv[i]!=NULL){
	if(pointer->flags&ARGC)
	 *((char **)pointer->var2)=argv[i];
	else{
	 if(pointer->flags&ARGN)
	  *((int *)pointer->var2)=atol(argv[i]);
	 else
	  if(pointer->flags&ARGF)
	   *((float *)pointer->var2)=(float)atof(argv[i]);
	}
       }else
	i--;
      }
      break;
     }
     pointer++;
    }
   }else
    if(freeoptcount<MAXFREEOPTS)
     freeopts[freeoptcount++]=&argv[i][0];
  }
 }else
  disphelp=1;
 if(disphelp)
  display_help();
}

void opt_reset_variables(acv_fileinfo *af_in,acv_fileinfo *af_out)
{
 memset((void *)af_in,0,sizeof(struct acv_fileinfo));
 memset((void *)af_out,0,sizeof(struct acv_fileinfo));
 opt_normlimit=0.1f;
}

void opt_accept_filenames(acv_fileinfo *af_in,acv_fileinfo *af_out)
{
 if(freeopts[1])
  af_in->filename=freeopts[1];
 else
  display_help();

 if(freeopts[2]){
  af_out->filename=freeopts[2];
 }else
  if(opt_conv_functions&CF_PASS2) // output filename is needed for conversions
   display_help();
}

void opt_accept_variables(acv_fileinfo *af_in,acv_fileinfo *af_out)
{
 af_in->filemode=af_out->filemode=opt_conv_functions;
 if(af_in->filename)
  af_in->filemode|=CF_OUTPUT;
 if(af_out->filename)
  af_out->filemode|=CF_OUTPUT;

 af_out->bitrate=opt_bitrate*1000;

 if(opt_outbits>PCM_MAX_BITS)
  opt_outbits=PCM_MAX_BITS;
 af_out->databits=af_out->scalebits=opt_outbits;
 af_out->freq    =opt_outfreq;
 af_out->channels=opt_outchannels;

 if(tag_artist)
  af_out->tag_artist=tag_artist;
 if(tag_title)
  af_out->tag_title =tag_title;
 af_out->tag_album =tag_album;
 af_out->tag_date  =tag_date;
 af_out->tag_genre =tag_genre;
 af_out->tag_comment=tag_comment;

 if(opt_normdb)
  cv_normdb=(float)acv_fromdB(opt_normdb);
 else
  cv_normdb=(float)acv_fromdB(192.0f); // max. dynamic (32 pcm bits)
 cv_normlimit=(float)acv_fromdB(opt_normlimit);

#ifndef _WIN32
 if(opt_charset)
  convert_set_charset(opt_charset);
#endif
}

static void display_help(void)
{
 fprintf(stdout,"AudioCV v%s for %s by PDSoft (http://mpxplay.cjb.net)\n",PRG_VERSION,OS_TYPE);
 fprintf(stdout,"Usage: AudioCV.exe [options] infile outfile\n");
 fprintf(stdout,"\nImplemented functions:\n");
 fprintf(stdout," - Ogg file encoding (audiocv.exe infile.wav outfile.ogg)\n");
 fprintf(stdout," - Ogg file decoding (audiocv.exe infile.ogg outfile.wav)\n");
 fprintf(stdout," - wav converter (audiocv.exe infile.wav outfile.wav)\n");

 fprintf(stdout,"\nOgg encoder options:\n");
 fprintf(stdout," -br NUM  : set bitrate (45-480 at 44.1kHz)(def.:128)\n");
 fprintf(stdout," -oeq NUM : set encoding quality instead of bitrate (0.0-1.0)(see readme.txt)\n");
 fprintf(stdout," -ocp     : use channel coupling (recommended/required for low bitrates)\n");
 fprintf(stdout," -ocb     : use constant bitrate (else VBR)(not recommended)\n");
 fprintf(stdout," -olp KHZ : set lowpass filter (cut off high sounds)(ie:-olp 16.5)\n");
 fprintf(stdout," -teu8    : use UTF-8 tag-text encoding\n");
#ifndef _WIN32
 fprintf(stdout," -tecp CPNAME : set codepage to UTF-8 (see readme.txt)\n");
#endif
 fprintf(stdout," -ta \"artist name\"\n");
 fprintf(stdout," -tt \"title of song\"\n");
 fprintf(stdout," -tl \"album name\"\n");
 fprintf(stdout," -td \"date\"\n");
 fprintf(stdout," -tg \"genre\"\n");
 fprintf(stdout," -tc \"comment\"\n");

 fprintf(stdout,"\nWave editor functions:\n");
 fprintf(stdout," -ct     : cut zero samples (from the begin and the end of file)\n");
 fprintf(stdout," -ctl NUM: cut samples below NUM sign (ie: -ctl 4)\n");
 fprintf(stdout," -mtl NUM: mute (clear) samples below NUM sign (ie: -mtl 7)\n");

 fprintf(stdout," -nr     : normalize sound\n");
 fprintf(stdout," -na     : analyze sound (only)\n");
 fprintf(stdout," -ndb F.F: modify sound amplitude with F.F dB (ie: -ndb 3.5)\n");
 fprintf(stdout," -ncl NUM: (number of) clips limit at normalization (ie: -ncl 3)\n");
 fprintf(stdout," -nrl F.F: normalization limit (ie: -nrl 0.8)\n");

 fprintf(stdout," -of NUM : output frequency (ie: 22050, 44100)\n");
 fprintf(stdout," -oc NUM : output channels (1 or 2)(1->2 or 2->1 only)\n");
 fprintf(stdout," -ob NUM : output bits/sample of wav files (8,16,20,24,32)\n");
 fprintf(stdout," -obf    : write 32-bit float wav files\n");
 fprintf(stdout," -sw     : swap channels (reverse stereo)\n");

 fprintf(stdout," -llo LST: use encoding list, create OGG files (see samples\\loadlist.lst)\n");
 fprintf(stdout," -llw LST: use encoding list, create WAV files\n");
 fprintf(stdout," -ll LST : use encoding list, don't create output (use for -na)\n");

 fprintf(stdout," -idel   : delete input file after conversion/encoding\n");
 fprintf(stdout," -iow    : overwrite all existent files (else writes error msg)\n");

 fprintf(stdout," -q      : do not display progress info (quiet)\n");

 exit(1);
}

//--------------------------------------------------------------------------
//list functions
static FILE *fp_list;
static char listline[400],outfilename[300]; // have to be static, not local!

int list_open(void)
{
 if(!loadlistname)
  return 0;
 if((fp_list=fopen(loadlistname,"rt"))==NULL)
  return 0;
 return 1;
}

void list_close(void)
{
 if(fp_list)
  fclose(fp_list);
}

static void list_slice(char **listparts[],char cutchars[])
{
 char *lastpart=&listline[0],*nextpart=NULL;
 int i,ccn=(int)strlen(cutchars);
 for(i=0;i<ccn;){
  *listparts[i]=lastpart;
  do{
   nextpart=strchr(lastpart,cutchars[i]);
   if(nextpart){
    *nextpart++=0;
    lastpart=nextpart;
   }
   i++;
  }while(!nextpart && i<ccn);
 }
}

static void list_create_outfilename(acv_fileinfo *af_in,acv_fileinfo *af_out)
{
 if(opt_outfiletype){
  strcpy(outfilename,af_in->filename);
  acv_change_extension_by_typenum(outfilename,opt_outfiletype);
  af_out->filename=&outfilename[0];
 }
}

int list_getnextentry(acv_fileinfo *af_in,acv_fileinfo *af_out)
{
 char **listparts[3];
 char sdisp[300];
 memset(listline,0,400);
 if(!feof(fp_list)){
  do{
   fgets(listline,399,fp_list);
  }while(!feof(fp_list) && (listline[0]=='#' || (strlen(listline)<2)));
  if(strlen(listline)>1){
   listparts[0]=&(af_in->filename);
   listparts[1]=&(af_out->tag_artist);
   listparts[2]=&(af_out->tag_title);
   list_slice(listparts," :\n");// slice listline at ' ',':' and '\n'
   if(!af_out->tag_title){
    af_out->tag_title=af_out->tag_artist;
    af_out->tag_artist=0;
   }
   list_create_outfilename(af_in,af_out);
   if(!dispquiet){
    sprintf(sdisp,"%s->%s  %s:%s",
	 (af_in->filename)?    af_in->filename:"(null)",
	 (af_out->filename)?   af_out->filename:"(null)",
	 (af_out->tag_artist)? af_out->tag_artist:"",
	 (af_out->tag_title)?  af_out->tag_title:"");
    fprintf(stderr,"%1.78s\n",sdisp);
   }
   return 1;
  }
 }
 return 0;
}
